/*** ---------------------------------------------------------------------------------------
 * In combination with the MFRC522 library (https://github.com/miguelbalboa/rfid )
 * this program sends the UiD of the RFID card to the serial port with 9600 baud.
 * 
 *  
 * Typical pin layout used - MFRC522:
 * -----------------------------------------------------------------------------------------
 *             MFRC522      Arduino       Arduino   Arduino    Arduino          Arduino
 *             Reader/PCD   Uno           Mega      Nano v3    Leonardo/Micro   Pro Micro
 * Signal      Pin          Pin           Pin       Pin        Pin              Pin
 * -----------------------------------------------------------------------------------------
 * RST/Reset   RST          9             5         D9         RESET/ICSP-5     RST
 * SPI SS      SDA(SS)      10            53        D10        5                10
 * SPI MOSI    MOSI         11 / ICSP-4   51        D11        ICSP-4           16
 * SPI MISO    MISO         12 / ICSP-1   50        D12        ICSP-1           14
 * SPI SCK     SCK          13 / ICSP-3   52        D13        ICSP-3           15
 * 
 */

#include <SPI.h>
#include <MFRC522.h>
#include <EEPROM.h>

#define RST_PIN         9           /* Configurable, see typical pin layout above*/
#define SS_PIN         10           /* Configurable, see typical pin layout above*/                                  
#define UID_LEN         7


MFRC522 mfrc522(SS_PIN, RST_PIN);         // Create MFRC522 instance.
MFRC522::MIFARE_Key key;

// Global variables
uint8_t oldUid[UID_LEN] = {0};           // stores the old UID 
unsigned long ResetUid;                  // stores the time moment when the old UID was stored.



void dump_byte_array(byte *buffer, byte bufferSize)                 // Only for test purposes
{
    for (byte i = 0; i < bufferSize; i++) {
        Serial.print(buffer[i] < 0x10 ? " 0" : " ");
        Serial.print(buffer[i]);
    }
}


/*** routine to send the UiD including STX, checksum and ETX to Serial.*/
void send_byte_array_incl_ChkSum(byte *buffer, byte bufferSize) 
{
  byte i=1, Checksum=0 ;
      
    Serial.write(0x02);                                              // STX (02h). You must send STX as a byte, only then STX is recognized as a command
    
    for (i = 1; i < 6; i++)                                          // Remove first and last byte, we can only manage messages with 5 bytes of data
    {                                                                                 
      if (i==1) Checksum  = buffer[i];
      else      Checksum ^= buffer[i];   
      if (buffer[i] < 0x10) Serial.print (0);                        // HEX value is always 2 bytes
      Serial.print(buffer[i],HEX);                                   // send information
    }
   if (Checksum < 0x10) Serial.print (0);                            // HEX value is always 2 bytes
   Serial.println(Checksum,HEX);                                     // send checksum
   Serial.write(0x03);                                               // ETX (03h). You must send ETX as a byte, only then ETX is recognized as a command
}

/******************************************
 * Function used to compare two RFID UIDs
 ******************************************/

bool compareUid(byte *buffer1, byte *buffer2, byte bufferSize) 
{
    for (byte i = 0; i < bufferSize; i++) 
    {
        if (buffer1[i] != buffer2[i]) 
           return false;                                              // if only one byte differs then leave the function and return "false"
    }   
    return true;   
}

/*****************************************
 * Function used to copy one RFID UID to another one
 *****************************************/
void copyUid (byte *buffIn, byte *buffOut, byte bufferSize)
{
    for (byte i = 0; i < bufferSize; i++) 
      {
        buffOut[i] = buffIn[i];
      }
    
    if(bufferSize < UID_LEN)
      {
       for (byte i = bufferSize; i < UID_LEN; i++) 
       {
           buffOut[i] = 0;
       }    
    }
    ResetUid = millis() + 2000;                                    // record the time, after 2 seconds the main loop() releases the blockade on the same rfid ID
 
}

  
/*************************************************************************************************************************************
 * Initialize.
 *************************************************************************************************************************************/
void setup() {
    uint32_t uiStartTimer;
    uint16_t uiElapsedDelay;
    uint16_t uiSerialOKDelay = 1000;
    
    Serial.begin(9600);                                             // Initialize serial communications on 9600 baud
    uiStartTimer = millis();
    do{                                                             //wait for the serial interface, but maximal 1 second.
        uiElapsedDelay = millis() - uiStartTimer;
    } while ((!Serial) && (uiElapsedDelay < uiSerialOKDelay));      // Do nothing if no serial port is opened (added for Arduinos based on ATMEGA32U4)
   
    SPI.begin();                                                    // Init SPI bus
    mfrc522.PCD_Init();                                             // Init MFRC522 card

    
 
}

/*************************************************************************************************************************************
 * Main loop.
 *************************************************************************************************************************************/
void loop() 
{

  if ( mfrc522.PICC_IsNewCardPresent() && mfrc522.PICC_ReadCardSerial())
    {  
        if ( ! compareUid( mfrc522.uid.uidByte, oldUid, mfrc522.uid.size))           // if the card-ID differs from the previous card-ID
          {     
           //  Serial.print(F("Card UID:"));
           //  dump_byte_array(mfrc522.uid.uidByte, mfrc522.uid.size);               // Card-Id in decimal figures, only for test
           //  Serial.println();
          
            send_byte_array_incl_ChkSum (mfrc522.uid.uidByte, mfrc522.uid.size);     // send card-ID (5 words) to the RS232 port in HEX, including Checksum
           
            copyUid ( mfrc522.uid.uidByte, oldUid, mfrc522.uid.size);                // fill oldUid with the latest card-ID
         }
       if (ResetUid < millis())  
           {
            oldUid[0] =0;              // Relaese, after 2 seconds, the blockade on the same rfid ID
           }
 
                 
    }       




}

